#Requires -Version 3.0
# =============================================================================================
# This script creates a html report of vSphere NTP settings.

# AUTHOR		Thomas Franke / sepago GmbH
# Blog			https://www.sepago.de/thomasf
# LASTEDIT		24.11.2014
# =============================================================================================

Set-StrictMode -Version Latest
Clear-Host

$VCenterServerName		= '<vCenter FQDN or IP>'
$UserName				= '<Domain\vCenter UserName>'

$AdvancedSettingName 	= 'SyncTimeWithHost'
	
$ReportTitle			= 'vSphere NTP Configuration'
$OutputFileName			= 'reports\Report-NtpConfig.html'

# =============================================================================================

Push-Location $(Split-Path $Script:MyInvocation.MyCommand.Path)

. .\include\Export-HtmlReport.ps1
. .\include\Get-PSCredential.ps1
Add-Pssnapin 'VMware.VimAutomation.Core' -ErrorAction SilentlyContinue

Try
{
	Disconnect-VIServer * -Force -Confirm:$False
}
Catch
{
	# nothing to do - no open connections!
}

$PSCredential = Get-PSCredential $UserName
Connect-VIserver -Server $VCenterServerName -Credential $PSCredential

$Property	= @()
$VMHosts	= Get-VMHost | Sort Name

$VMHosts | ForEach {

	$ofs = ','	# Separator for string conversions
	[string]$DnsAddress	= $_ | Get-VMHostNetwork | Select -Expandproperty DnsAddress
	[string]$NtpServer	= $_ | Get-VmHostNtpServer
	$NtpService			= $_ | Get-VmHostService | Where {$_.key -eq 'ntpd'}
	$NtpServicePolicy	= $NtpService.Policy.Replace('on','start with host').Replace('off','manual')
	$NtpFirewall		= $_ | Get-VmHostFirewallException | Where {$_.Name -eq 'NTP-Client'}
	
	$Setting = [ordered]@{
							HostName						= $_.Name
							DnsAddress						= $DnsAddress
							NtpServer						= $NtpServer
							NtpService_Running				= $NtpService.Running
							NtpService_Policy				= $NtpServicePolicy
							NtpFirewall_Exception_Enabled	= $NtpFirewall.Enabled
	}
	$Property += New-Object -TypeName PSObject -Property $Setting
}

$Property1	= @()
$Property2	= @()
$VMs		= Get-VM | Sort Name

$VMs | ForEach {

	$AdvancedSetting = @()
	$AdvancedSetting = $_.ExtensionData.Config.Tools | Select $AdvancedSettingName
	
	If ($AdvancedSetting)
	{
		$Setting = [ordered]@{
								VmName			= $_.Name
								ParameterName	= $AdvancedSettingName
								Value			= $AdvancedSetting.$AdvancedSettingName
		}
		
		If ($AdvancedSetting.$AdvancedSettingName -eq $True)
		{
			$Property1 += New-Object -TypeName PSObject -Property $Setting
		}
		Else
		{
			$Property2 += New-Object -TypeName PSObject -Property $Setting
		}		
	}
}

$FooterHash = [ordered]@{
							ScriptName		= $Script:MyInvocation.MyCommand.Name
							Author			= "thomas.franke@sepago.de"
							ScriptInvokedBy	= "$env:userdomain\$env:username"
							ExecutionDate	= Get-Date -Format yyyy-MM-dd
							ExecutionTime	= Get-Date -Format HH:mm
}
$FooterObject = New-Object -TypeName PSObject -Property $FooterHash

$InputObject =	@{
					Title		= "ESXi Host NTP Configuration"
					Object		= $Property
				},
				@{
					Title		= "VMs with TimeSync Enabled"
					Object		= $Property1
				},
				@{
					Title		= "VMs with TimeSync Disabled"
					Object		= $Property2
				},
				@{
					Object		= $FooterObject
				}

$InputObject | Export-HtmlReport -ReportTitle $ReportTitle -OutputFile $OutputFileName | Invoke-Item

Disconnect-VIserver -Server $VCenterServerName -Force -Confirm:$False
Pop-Location
