#Requires -Version 3.0
# =============================================================================================
# This script creates a html report of ESXi NIC driver versions.
# Only hosts with ConnectionState 'Connected' or 'Maintenance' can be processed.

# AUTHOR		Thomas Franke / sepago GmbH
# Blog			https://www.sepago.de/thomasf
# LASTEDIT		12.09.2014
# =============================================================================================

Set-StrictMode -Version Latest
Clear-Host

$VCenterServerName		= '<vCenter FQDN or IP>'
$UserName				= '<Domain\vCenter UserName>'

$ReportTitle			= 'VMware ESXi NICs and Drivers'
$OutputFileName			= 'reports\Report-VMHostNicDrivers.html'

# =============================================================================================

Push-Location $(Split-Path $Script:MyInvocation.MyCommand.Path)

. .\include\Export-HtmlReport.ps1
. .\include\Get-PSCredential.ps1
Add-Pssnapin 'VMware.VimAutomation.Core' -ErrorAction SilentlyContinue

Try
{
	Disconnect-VIServer * -Force -Confirm:$False
}
Catch
{
	# nothing to do - no open connections!
}

$PSCredential = Get-PSCredential $UserName
Connect-VIserver -Server $VCenterServerName -Credential $PSCredential

# Set language for current vcenter session
$ServiceInstance    = Get-View ServiceInstance
$SessionManager     = Get-View $ServiceInstance.Content.SessionManager
$SessionManager.SetLocale("en_US")

$VMHosts	= Get-VMHost | Where {($_.ConnectionState -eq 'Connected') -or ($_.ConnectionState -eq 'Maintenance')} | Sort Name

$Property = @()
ForEach ($VMHost in $VMHosts) {
	$Setting = @()
	$HostView = $VMHost | Get-View
	$HostCli = $VMHost | Get-EsxCli

	$HostNics = Get-VMHostNetworkAdapter -VMHost $VMHost | Where {$_.Name -like "vmnic*"} | ForEach {
		$NicName	= $_.Name
		$NicPciId	= $_.ExtensionData.Pci
		$PciDevice	= $HostView.Hardware.PciDevice | Where {$_.Id -eq $NicPciId}
		$NicDriver	= $HostCli.Network.Nic.get($NicName).DriverInfo

		$Setting 	= [ordered]@{
								HostName			= $VMHost.Name
								VmNic				= $_
								PciId				= $_.ExtensionData.Pci
								DeviceName			= $PciDevice.DeviceName
								VendorName			= $PciDevice.VendorName
								NicDriver			= $NicDriver.Driver
								NicDriverVersion	= $NicDriver.Version
		}
		$Property += New-Object -TypeName PSObject -Property $Setting
	}
}

$FooterHash = [ordered]@{
							ScriptName		= $Script:MyInvocation.MyCommand.Name
							Author			= "thomas.franke@sepago.de"
							ScriptInvokedBy	= "$env:userdomain\$env:username"
							ExecutionDate	= Get-Date -Format yyyy-MM-dd
							ExecutionTime	= Get-Date -Format HH:mm
}
$FooterObject = New-Object -TypeName PSObject -Property $FooterHash

$InputObject =	@{
					Title		= 'VMware ESXi NICs and Drivers'
					Object		= $Property
				},
				@{
					Object		= $FooterObject
				}

$InputObject | Export-HtmlReport -ReportTitle $ReportTitle -OutputFile $OutputFileName | Invoke-Item

Disconnect-VIserver -Server $VCenterServerName -Force -Confirm:$False
Pop-Location
