#Requires -Version 3.0
# =============================================================================================
# This script creates a html report of the VM memory overhead

# AUTHOR		Thomas Franke / sepago GmbH
# Blog			https://www.sepago.de/thomasf
# LASTEDIT		11.01.2015
# =============================================================================================

Set-StrictMode -Version Latest
Clear-Host

$VCenterServerName		= '<vCenter FQDN or IP>'
$UserName				= '<Domain\vCenter UserName>'

$ReportTitle			= 'VMware VM memory overhead'
$OutputFileName			= 'reports\Report-VmMemoryOverhead.html'

# =============================================================================================

Push-Location $(Split-Path $Script:MyInvocation.MyCommand.Path)

. .\include\Export-HtmlReport.ps1
. .\include\Get-PSCredential.ps1
Add-Pssnapin 'VMware.VimAutomation.Core' -ErrorAction SilentlyContinue

Try
{
	Disconnect-VIServer * -Force -Confirm:$False
}
Catch
{
	# nothing to do - no open connections!
}

$PSCredential = Get-PSCredential $UserName
Connect-VIserver -Server $VCenterServerName -Credential $PSCredential

# Set vCenter connection language setting
$ServiceInstance	= Get-View ServiceInstance
$SessionManager		= Get-View $ServiceInstance.Content.SessionManager
$SessionManager.SetLocale('en_US')

$Property = @()
$VMHosts = Get-View -ViewType HostSystem | Sort Name
ForEach ($VM in (Get-VM))
{
	$VMHost = $VMHosts | Where {$_.Name -eq $VM.VMHost}
	$VMOverheadMB = [math]::Round(($VMHost.QueryMemoryOverheadEx($VM.ExtensionData.config) / 1MB), 2)
	
	$Setting = [ordered]@{
							VmHostName			= $VMHost.Name
							VmName				= $VM.Name
							VmNumCpu			= $VM.NumCpu
							VmMemoryGB			= $VM.MemoryGB
							MemoryOverheadMB	= $VMOverheadMB
	}
	$Property += New-Object -TypeName PSObject -Property $Setting
}

$Property1 = @()
ForEach ($VMHost in $VMHosts)
{
	$VMs = $Property | Where {$VMHost.Name -eq $_.VmHostName}
	If ($VMs -ne $Null)
	{
		$VmMemorySumGB 			= ($VMs | Measure-Object -Property VmMemoryGB -Sum).Sum
		$MemoryOverHeadSumGB	= [math]::Round((($VMs | Measure-Object -Property MemoryOverheadMB -Sum).Sum) / 1024, 2)
	}
	Else
	{
		$VmMemorySumGB 			= 0
		$MemoryOverHeadSumGB	= 0

	}
		$HostMemoryGB			= [math]::Round($VMHost.Hardware.MemorySize / 1GB, 2)
		$UsedHostMemoryGB		= [math]::Round($VmMemorySumGB + $MemoryOverHeadSumGB, 2)
		$Property1 +=  '' | Select @{Name = 'HostName'; Expression = {$VMHost.Name}}, @{Name = 'HostMemoryGB'; Expression = {$HostMemoryGB}}, @{Name = 'VmMemorySumGB'; Expression = {$VmMemorySumGB}}, @{Name = 'MemoryOverHeadSumGB'; Expression = {$MemoryOverHeadSumGB}}, @{Name = 'TotalVMMemoryGB'; Expression = {$UsedHostMemoryGB}}
}

$FooterHash = [ordered]@{
							ScriptName		= $Script:MyInvocation.MyCommand.Name
							Author			= "thomas.franke@sepago.de"
							ScriptInvokedBy	= "$env:userdomain\$env:username"
							ExecutionDate	= Get-Date -Format yyyy-MM-dd
							ExecutionTime	= Get-Date -Format HH:mm
}
$FooterObject = New-Object -TypeName PSObject -Property $FooterHash

$InputObject =	@{
					Title		= "VM Memory Overhead"
					Object		= $Property | Sort VmName
				},
				@{
					Title		= "Host Memory"
					Description	= "The TotalVMMemoryGB is the amount of RAM needed for the VMs including memory overhead. The required physical RAM will usually be lower."
					Object		= $Property1
				},
				@{
					Object		= $FooterObject
				}
				
$InputObject | Export-HtmlReport -ReportTitle $ReportTitle -OutputFile $OutputFileName | Invoke-Item

Disconnect-VIserver -Server $VCenterServerName -Force -Confirm:$False
Pop-Location
